﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.Drawing;
using System.Drawing.Imaging;
using System.Collections;
using System.IO;
using System.Threading;

using com.google.zxing;
using com.google.zxing.qrcode;
using com.google.zxing.common;
using com.google.zxing.qrcode.decoder;
using com.google.zxing.multi;

using IEC16022Sharp;

namespace QRCode
{
    public class DecoderThreadHandler
    {
        private ManualResetEvent _doneEvent;
        private ManualResetEvent[] _Events;
        public int ResultThreadIndex = -1;

        public DecoderThreadHandler(ManualResetEvent[] Events, ManualResetEvent doneEvent)
        {
            _Events = Events;
            _doneEvent = doneEvent;
        }

        public void WaitAllEvt()
        {
            WaitHandle.WaitAll(_Events);
            _doneEvent.Set();
        }

        public void WaitAnyEvt()
        {
            ResultThreadIndex = WaitHandle.WaitAny(_Events);
            _doneEvent.Set();
        }
    }

    public static class multiDecoder
    {
        private static List<BarcodeFormat> _fmtList;
        private static ManualResetEvent[] _emptyEvents;
        private static ManualResetEvent[] _resultEvents;
        private static ManualResetEvent[] _ThreadHandlersDoneEvents;
        private static Thread[] _myDecoderThreads;
        private static tDecoder[] _myDecoders;

        private static Thread[] _ThreadHandlers;
        private static DecoderThreadHandler _ResultHandler;
        private static DecoderThreadHandler _EmptyHandler;

        public static void GenFmtList()
        {
            _fmtList = new List<BarcodeFormat>();
            _fmtList.Add(BarcodeFormat.QR_CODE);
            _fmtList.Add(BarcodeFormat.CODE_128);
            _fmtList.Add(BarcodeFormat.CODE_39);
            _fmtList.Add(BarcodeFormat.DATAMATRIX);
            _fmtList.Add(BarcodeFormat.EAN_13);
            _fmtList.Add(BarcodeFormat.EAN_8);
            _fmtList.Add(BarcodeFormat.ITF);
            _fmtList.Add(BarcodeFormat.PDF417);
            _fmtList.Add(BarcodeFormat.UPC_A);
            _fmtList.Add(BarcodeFormat.UPC_E);
        }

        public static void initEvents()
        {
            //Decoder events section
            _emptyEvents = new ManualResetEvent[_fmtList.Count];
            _resultEvents = new ManualResetEvent[_fmtList.Count];

            //Thread handlers events section
            _ThreadHandlersDoneEvents = new ManualResetEvent[2];

            //Initialise Thread Handler Events
            _ThreadHandlersDoneEvents[0] = new ManualResetEvent(false);
            _ThreadHandlersDoneEvents[1] = new ManualResetEvent(false);
        }

        public static void startDecoderThreads()
        {
            for (int i = 0; i < _myDecoders.Length; i++)
            {
                _myDecoderThreads[i].Start();
            }
        }

        public static void setupDecoderThreads()
        {
            _myDecoderThreads = new Thread[_myDecoders.Length];
            for (int i = 0; i < _myDecoders.Length; i++)
            {
                _myDecoderThreads[i] = new Thread(new ThreadStart(_myDecoders[i].decode));
            }
        }

        public static void cleanupThreadHandlerThreads()
        {
            //Abort our Handlers
            _ThreadHandlers[0].Abort();
            _ThreadHandlers[1].Abort();

            _ThreadHandlers[0] = null;
            _ThreadHandlers[1] = null;
        }

        public static void startThreadHandlerThreads()
        {
            _ThreadHandlers = new Thread[2];

            _EmptyHandler = new DecoderThreadHandler(_emptyEvents, _ThreadHandlersDoneEvents[0]);
            _ResultHandler = new DecoderThreadHandler(_resultEvents, _ThreadHandlersDoneEvents[1]);

            _ThreadHandlers[0] = new Thread(new ThreadStart(_EmptyHandler.WaitAllEvt));
            _ThreadHandlers[1] = new Thread(new ThreadStart(_ResultHandler.WaitAnyEvt));

            _ThreadHandlers[0].Start();
            _ThreadHandlers[1].Start();
        }

        public static void setupDecoders(Bitmap img)
        {
            _myDecoders = new tDecoder[_fmtList.Count];
            for (int i = 0; i < _fmtList.Count; i++)
            {
                _emptyEvents[i] = new ManualResetEvent(false);
                _resultEvents[i] = new ManualResetEvent(false);
                _myDecoders[i] = new tDecoder((Bitmap)img.Clone(), _fmtList[i],  _emptyEvents[i], _resultEvents[i]);
            }
        }

        public static List<Result> getResult()
        {

            _ThreadHandlers[1].Join();
            int resultThreadIdx = _ResultHandler.ResultThreadIndex;

            cleanupThreadHandlerThreads();

            //Abort old threads
            for (int i = 0; i < _myDecoders.Length; i++)
            {
                if (i != resultThreadIdx) _myDecoderThreads[i].Abort();
            }

            _myDecoderThreads[resultThreadIdx].Join();
            tDecoder tDec = _myDecoders[resultThreadIdx];

            List<Result> newList = new List<Result>(tDec.results);

            _myDecoderThreads[resultThreadIdx].Abort();

            return newList;
        }

        public static List<Result> decode(Bitmap img)
        {

            GenFmtList();
            initEvents();
            setupDecoders(img);
            setupDecoderThreads();
            startThreadHandlerThreads();
            startDecoderThreads();

            int index = WaitHandle.WaitAny(_ThreadHandlersDoneEvents);

            //This means we had to wait for all the script to finish and there were no results...
            if (index == 0)
            {
                cleanupThreadHandlerThreads();
                return new List<Result>();
            }

            return getResult();
        }
    }

    class tEncoder
    {
        private string _data;
        private Hashtable _HINTS;

        public tEncoder(string data)
        {
            _data=data;
        }

        public Bitmap encode()
        {
            return encode(130);
        }

        public Bitmap encode(int dim)
        {
            return encode(dim, ErrorCorrectionLevel.M);
        }

        public Bitmap encode(int dim, ErrorCorrectionLevel ECL)
        {
            return encode(dim, dim, ECL);
        }

        public Bitmap encode(int width, int height, ErrorCorrectionLevel ECL)
        {
            _HINTS = new Hashtable();
            _HINTS.Add(EncodeHintType.ERROR_CORRECTION, ECL);

            MultiFormatWriter writer = new MultiFormatWriter();
            com.google.zxing.common.ByteMatrix matrix = writer.encode(_data, BarcodeFormat.QR_CODE, width, height, _HINTS);

            Bitmap img = new Bitmap(width, height);

            for (int y = 0; y < matrix.Height; ++y)
            {
                for (int x = 0; x < matrix.Width; ++x)
                {
                    //Find the colour of the dot 
                    if (matrix.get_Renamed(x, y) == -1)
                    {
                        img.SetPixel(x, y, Color.White);
                    }
                    else
                    {
                        img.SetPixel(x, y, Color.Black);
                    }
                }
            }

            return img;
        }
    }

    class tDecoder
    {
        private ManualResetEvent _emptyEvent;
        private ManualResetEvent _resultsEvent;

        private Hashtable _HINTS;
        private Reader _reader;
        public List<Result> results;
        private List<string> _errors;
        private LuminanceSource _lsrc;

        private void init(){
            _errors = new List<string>();
            _reader = new MultiFormatReader();
            results = new List<Result>();
            _HINTS = new Hashtable();
        }

        private void lmImage(Bitmap bmImage)
        {
            _lsrc = new RGBLuminanceSource(bmImage, bmImage.Width, bmImage.Height);
        }

        private void MultiDecode(BinaryBitmap bbmap)
        {
            try
            {
                // Look for multiple barcodes
                MultipleBarcodeReader multiReader = new GenericMultipleBarcodeReader(_reader);
                Result[] theResults = multiReader.decodeMultiple(bbmap, _HINTS);
                if (theResults != null)
                {
                    results.AddRange(theResults.ToList<Result>());
                }
            }
            catch (ReaderException re)
            {
                _errors.Add(re.Message);
            }
            catch (Exception ex)
            {
                _errors.Add(ex.Message);
            }
        }

        private void SingleDecode(BinaryBitmap bbmap)
        {
            try
            {
                // Look for normal barcode in photo
                Result theResult = _reader.decode(bbmap, _HINTS);
                if (theResult != null)
                {
                    results.Add(theResult);
                }
            }
            catch (ReaderException re)
            {
                _errors.Add(re.Message);
            }
            catch (Exception ex)
            {
                _errors.Add(ex.Message);
            }
        }

        private void PureDecode(BinaryBitmap bbmap)
        {
            try
            {
                // Look for pure barcode
                Result theResult = _reader.decode(bbmap, _HINTS);
                if (theResult != null)
                {
                    results.Add(theResult);
                }
            }
            catch (ReaderException re)
            {
                _errors.Add(re.Message);
            }
            catch (Exception ex)
            {
                _errors.Add(ex.Message);
            }
        }

        public tDecoder(Bitmap bmImage, BarcodeFormat MyFormat, ManualResetEvent doneEvent,ManualResetEvent resultsEvent)
        {
            constructor(bmImage, MyFormat, doneEvent, resultsEvent, true);
        }

        public tDecoder(Bitmap bmImage, BarcodeFormat MyFormat, ManualResetEvent doneEvent,ManualResetEvent resultsEvent, bool tryhard)
        {
            constructor(bmImage, MyFormat, doneEvent,resultsEvent, tryhard);
        }

        private void constructor(Bitmap bmImage, BarcodeFormat MyFormat, ManualResetEvent doneEvent, ManualResetEvent resultsEvent, bool tryhard)
        {
            _emptyEvent = doneEvent;
            _resultsEvent = resultsEvent;

            init();
            lmImage(bmImage);

            ArrayList myList = new ArrayList();
            myList.Add(MyFormat);

            if (tryhard) _HINTS.Add(DecodeHintType.TRY_HARDER, true);
            _HINTS.Add(DecodeHintType.POSSIBLE_FORMATS, myList);
        }

        public void decode()
        {

            if (results.Count == 0) MultiDecode(new BinaryBitmap(new GlobalHistogramBinarizer(_lsrc)));
            if (results.Count == 0) MultiDecode(new BinaryBitmap(new HybridBinarizer(_lsrc)));
            if (results.Count == 0) SingleDecode(new BinaryBitmap(new GlobalHistogramBinarizer(_lsrc)));
            if (results.Count == 0) SingleDecode(new BinaryBitmap(new HybridBinarizer(_lsrc)));
            
            //Now try pure barcodes just in case
            _HINTS.Add(DecodeHintType.PURE_BARCODE, true);
            if (results.Count == 0) PureDecode(new BinaryBitmap(new GlobalHistogramBinarizer(_lsrc)));
            if (results.Count == 0) PureDecode(new BinaryBitmap(new HybridBinarizer(_lsrc)));

            if (results.Count != 0) _resultsEvent.Set();
            if (results.Count == 0) _emptyEvent.Set();
        }
    }

    class Program
    {
        static void Main(string[] args)
        {

            //tEncoder myCoder = new tEncoder("Hello");
            //string pathQR = @"C:\Users\Home\Pictures\QR.jpg";
            //myCoder.encode().Save(pathQR, ImageFormat.Jpeg);

            //DataMatrix dm1 = new DataMatrix(
            //        "Lorem ipsum dolor sit amet, consectetur adipisicing elit, " +
            //        "sed do eiusmod tempor incididunt ut labore et dolore magna aliqua dsfksdfsdjf sjdfklsdjfklsdjflk asdhfl ;asdh;ljsdh ;jksdhafjkasdhfjk;hsdkj;fhjks ;dhfjkhsjkdfhjkasdhfjkhsadf");
            //DMImgUtility.SimpleResizeBmp(dm1.Image, 10, 0).Save(pathQR, ImageFormat.Jpeg);
            //dm1.Image.Save(@"C:\Users\Home\Pictures\QR1.jpg", ImageFormat.Jpeg);

            if (args.Length > 0)
            {

                string filePath = args[0];

                if (File.Exists(filePath))
                {

                    List<Result> myResults = multiDecoder.decode(new Bitmap(filePath));

                    Console.WriteLine("All calculations are complete.");
                    Console.WriteLine("");

                    foreach (Result res in myResults)
                    {
                        Console.WriteLine(res.Text);
                    }

                    Console.ReadKey();
                }
            }
        }
    }
}
